class RainforestApp {
    constructor() {
        this.selectedAnimal = '';
        
        this.init();
    }

    init() {
        this.cacheDom();
        this.setupNavigation();
        this.renderAnimals();
        this.setupAnimalSelection();
        this.setupEnterKey();
    }

    cacheDom() {
        this.homeView = document.getElementById('home-view');
        this.chatView = document.getElementById('chat-view');
        this.backBtn = document.getElementById('back-to-home');
        this.selectedAnimalSpan = document.querySelector('.selected-animal');
        this.textQuestion = document.getElementById('text-question');
        this.sendBtn = document.getElementById('send-btn');
        this.answerArea = document.getElementById('answer-area');
    }

    setupNavigation() {
        if (this.backBtn) {
            this.backBtn.addEventListener('click', () => {
                this.navigateToHome();
            });
        }
    }

    navigateToChat() {
        if (this.homeView && this.chatView) {
            this.homeView.style.display = 'none';
            this.chatView.style.display = 'block';
        }
    }

    navigateToHome() {
        this.selectedAnimal = '';
        this.selectedAnimalSpan.textContent = '—';
        this.textQuestion.value = '';
        this.textQuestion.disabled = true;
        this.sendBtn.disabled = true;
        document.getElementById('suggested-questions').innerHTML = '<div class="suggestion-chip">请先选择一种动物</div>';
        this.answerArea.innerHTML = '<p style="color: #999; text-align: center;">AI回答将显示在这里</p>';
        
        document.querySelectorAll('.animal-card').forEach(card => {
            card.classList.remove('selected');
        });
        
        if (this.homeView && this.chatView) {
            this.chatView.style.display = 'none';
            this.homeView.style.display = 'block';
            window.scrollTo({ top: 0, behavior: 'smooth' });
        }
    }

    renderAnimals() {
        const animalGrid = document.querySelector('.animal-grid');
        animalGrid.innerHTML = ANIMALS_DATA.map(animal => `
            <div class="animal-card" data-animal="${animal.name}">
                <img src="${animal.image}" alt="${animal.alt}">
                <div class="animal-name">${animal.name}</div>
            </div>
        `).join('');
    }

    setupAnimalSelection() {
        document.addEventListener('click', (e) => {
            const animalCard = e.target.closest('.animal-card');
            if (!animalCard) return;

            document.querySelectorAll('.animal-card').forEach(card => {
                card.classList.remove('selected');
            });
            
            animalCard.classList.add('selected');
            this.selectedAnimal = animalCard.getAttribute('data-animal');
            this.selectedAnimalSpan.textContent = this.selectedAnimal;
            
            this.textQuestion.disabled = false;
            this.sendBtn.disabled = false;
            
            this.updateQuestionSuggestions();
            this.navigateToChat();
            this.scrollToQuestionArea();
        });
    }

    updateQuestionSuggestions() {
        const suggestions = QUESTION_SUGGESTIONS[this.selectedAnimal] || [];
        const suggestionsHTML = suggestions.map(q => 
            `<div class="suggestion-chip" onclick="app.handleSuggestionClick('${q.replace(/'/g, "\\'")}')">${q}</div>`
        ).join('');
        
        document.getElementById('suggested-questions').innerHTML = suggestionsHTML;
        
        // 添加建议问题的动画效果
        setTimeout(() => {
            const chips = document.querySelectorAll('.suggestion-chip');
            chips.forEach((chip, index) => {
                setTimeout(() => {
                    chip.style.opacity = '0';
                    chip.style.transform = 'translateY(20px)';
                    chip.style.transition = 'all 0.3s ease';
                    
                    setTimeout(() => {
                        chip.style.opacity = '1';
                        chip.style.transform = 'translateY(0)';
                    }, 50);
                }, index * 100);
            });
        }, 100);
    }

    handleSuggestionClick(question) {
        document.getElementById('text-question').value = question;
        this.askQuestion(question);
    }


    setupEnterKey() {
        document.getElementById('text-question').addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                this.askQuestion();
            }
        });
    }

    // 移除步骤指示器逻辑

    // 平滑滚动到问题输入区域
    scrollToQuestionArea() {
        const questionArea = document.querySelector('.question-area');
        if (questionArea) {
            // 添加高亮效果
            questionArea.classList.add('highlight');
            
            // 显示引导提示
            this.showGuideTip();
            
            // 平滑滚动到问题区域
            questionArea.scrollIntoView({ 
                behavior: 'smooth', 
                block: 'center' 
            });
            
            // 自动聚焦到输入框
            setTimeout(() => {
                document.getElementById('text-question').focus();
                // 移除高亮效果
                setTimeout(() => {
                    questionArea.classList.remove('highlight');
                }, 2000);
            }, 500);
        }
    }

    // 显示引导提示
    showGuideTip() {
        // 移除之前的提示
        const existingTip = document.querySelector('.guide-tip');
        if (existingTip) {
            existingTip.remove();
        }

        // 创建引导提示
        const tip = document.createElement('div');
        tip.className = 'guide-tip';
        tip.innerHTML = `
            <div class="guide-content">
                <span class="guide-icon">💡</span>
                <span class="guide-text">现在可以输入问题或点击下方建议问题开始对话！</span>
            </div>
        `;
        
        document.body.appendChild(tip);
        
        // 3秒后自动移除
        setTimeout(() => {
            if (tip.parentNode) {
                tip.remove();
            }
        }, 3000);
    }

    showError(message) {
        const answerArea = document.getElementById('answer-area');
        answerArea.innerHTML = `
            <div class="error-message">
                <p>${message}</p>
            </div>
        `;
    }

    async askQuestion(customQuestion = '') {
        const question = customQuestion || document.getElementById('text-question').value;
        
        if (!this.selectedAnimal) {
            this.showError('请先选择一种动物！');
            return;
        }
        
        if (!question.trim()) {
            this.showError('请输入问题！');
            return;
        }
        
        this.answerArea.innerHTML = '<div class="loading">🌿 AI正在思考中，请稍候...</div>';
        
        try {
            // 构建AI提示词
            const prompt = `请你扮演一只${this.selectedAnimal}，用生动有趣、简单易懂的语言回答一个小学生的问题。
问题：${question}

要求：
1. 回答要简短，最多8句话，总共不超过300字
2. 使用第一人称"我"来回答，体现动物个性
3. 语言要可爱、有童趣，适合小学生理解
4. 可以加入拟声词，比如"呼呼"、"沙沙"等
5. 确保科学知识准确但用孩子能懂的方式表达
6. 回答要体现这种动物的特点和生活习性`;

            await this.callStreamingAI(prompt, question);
            
        } catch (error) {
            console.error('API调用错误:', error);
            // 使用本地备用回答
            const fallbackAnswer = this.getFallbackAnswer(this.selectedAnimal, question);
            this.displayAnswer(fallbackAnswer, question);
            this.updateStepIndicator(3);
        }
    }

    // 直接调用火山方舟API
    async callStreamingAI(prompt, question) {
        const answerArea = document.getElementById('answer-area');
        const animal = ANIMALS_DATA.find(a => a.name === this.selectedAnimal);
        const avatarUrl = animal ? animal.image : '';
        
        // 显示流式回答区域
        answerArea.innerHTML = `
            <div class="answer-content">
                <div class="answer-header">
                    <img class="answer-avatar" src="${avatarUrl}" alt="${this.selectedAnimal}">
                    <div class="answer-animal">${this.selectedAnimal}回答：</div>
                </div>
                <div class="answer-text" id="streaming-answer"></div>
            </div>
        `;
        
        const streamingAnswerElement = document.getElementById('streaming-answer');
        let fullAnswer = '';
        
        try {
            // 生成动物特定的系统提示词
            const systemPrompt = this.generateAnimalSystemPrompt(this.selectedAnimal);
            
            const requestData = {
                model: API_CONFIG.volcano.model,
                messages: [
                    {
                        role: "system",
                        content: systemPrompt
                    },
                    {
                        role: "user",
                        content: prompt
                    }
                ],
                thinking: {
                    type: "disabled"
                },
                temperature: 0.7,
                max_tokens: 500,
                stream: true
            };
            
            const response = await fetch(API_CONFIG.volcano.apiUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${API_CONFIG.volcano.apiKey}`
                },
                body: JSON.stringify(requestData)
            });
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const reader = response.body.getReader();
            const decoder = new TextDecoder();
            
            while (true) {
                const { done, value } = await reader.read();
                
                if (done) break;
                
                const chunk = decoder.decode(value);
                const lines = chunk.split('\n');
                
                for (const line of lines) {
                    if (line.startsWith('data: ')) {
                        const data = line.slice(6);
                        if (data === '[DONE]') {
                            this.displayStreamingComplete(fullAnswer, question);
                            return;
                        }
                        
                        if (data.trim()) {
                            try {
                                const parsed = JSON.parse(data);
                                
                                if (parsed.choices && parsed.choices[0] && parsed.choices[0].delta && parsed.choices[0].delta.content) {
                                    const content = parsed.choices[0].delta.content;
                                    fullAnswer += content;
                                    streamingAnswerElement.textContent = fullAnswer;
                                    
                                    // 滚动到底部
                                    streamingAnswerElement.scrollTop = streamingAnswerElement.scrollHeight;
                                }
                            } catch (e) {
                                console.warn('解析流式数据错误:', e);
                            }
                        }
                    }
                }
            }
            
            // 如果没有通过[DONE]结束，也要显示完成状态
            if (fullAnswer) {
                this.displayStreamingComplete(fullAnswer, question);
            }
            
        } catch (error) {
            console.error('直接API调用错误:', error);
            throw error;
        }
    }

    // 显示流式回答完成状态
    displayStreamingComplete(answer, question) {
        const answerArea = document.getElementById('answer-area');
        const animal = ANIMALS_DATA.find(a => a.name === this.selectedAnimal);
        const avatarUrl = animal ? animal.image : '';
        
        answerArea.innerHTML = `
            <div class="answer-content">
                <div class="answer-header">
                    <img class="answer-avatar" src="${avatarUrl}" alt="${this.selectedAnimal}">
                    <div class="answer-animal">${this.selectedAnimal}回答：</div>
                </div>
                <div class="answer-text">${answer}</div>
            </div>
            <div class="controls">
                <button class="continue-btn" onclick="app.askAnotherQuestion()">
                    <span>❓</span> 继续提问
                </button>
            </div>
        `;
    }

    // 本地备用知识库查询
    getFallbackAnswer(animal, question) {
        const animalKnowledge = KNOWLEDGE_BASE[animal] || KNOWLEDGE_BASE["树懒"];
        const q = question.toLowerCase();
        
        // 关键词匹配
        const keywords = ['慢', '住', '层', '吃', '睡', '颜色', '飞', '蚂蚁', '舌头', '快', '手臂', '大', '腿'];
        
        for (const keyword of keywords) {
            if (q.includes(keyword) && animalKnowledge[keyword]) {
                return animalKnowledge[keyword];
            }
        }
        
        return animalKnowledge['default'];
    }

    // 显示答案
    displayAnswer(answer, question) {
        const answerArea = document.getElementById('answer-area');
        const animal = ANIMALS_DATA.find(a => a.name === this.selectedAnimal);
        const avatarUrl = animal ? animal.image : '';
        
        answerArea.innerHTML = `
            <div class="answer-content">
                <div class="answer-header">
                    <img class="answer-avatar" src="${avatarUrl}" alt="${this.selectedAnimal}">
                    <div class="answer-animal">${this.selectedAnimal}回答：</div>
                </div>
                <div class="answer-text">${answer}</div>
            </div>
            <div class="controls">
                <button class="continue-btn" onclick="app.askAnotherQuestion()">
                    <span>❓</span> 继续提问
                </button>
            </div>
        `;
    }


    // 继续提问
    askAnotherQuestion() {
        document.getElementById('text-question').value = '';
        document.getElementById('text-question').focus();
    }

    // 生成动物特定的系统提示词
    generateAnimalSystemPrompt(animal) {
        const animalPrompts = {
            "树懒": `你是一只生活在热带雨林冠层的树懒。你的特点是动作非常缓慢，一天要睡15-18小时，主要吃树叶和嫩芽。你住在高高的树上，很少下到地面。你的动作慢是为了节省能量，因为树叶的营养价值很低。请用第一人称'我'来回答，语言要可爱、有童趣，适合小学生理解。回答要简短，最多3句话，总共不超过100字。可以加入拟声词如'呼呼'、'沙沙'等。`,
            
            "五彩金刚鹦鹉": `你是一只五彩金刚鹦鹉，生活在热带雨林的露生层和冠层。你有着美丽的彩色羽毛，红色、蓝色、黄色都有。你爱吃水果、坚果和种子，用强壮的喙敲开坚硬的果壳。你会在雨林上空优雅地飞翔。请用第一人称'我'来回答，语言要可爱、有童趣，适合小学生理解。回答要简短，最多3句话，总共不超过100字。可以加入拟声词如'嘎嘎'、'呼呼'等。`,
            
            "食蚁兽": `你是一只食蚁兽，主要生活在地面上。你最爱吃蚂蚁，有60厘米长的舌头，上面有粘粘的唾液，可以轻松捕捉蚂蚁。你住在地表层，靠近蚂蚁巢穴的地方。请用第一人称'我'来回答，语言要可爱、有童趣，适合小学生理解。回答要简短，最多3句话，总共不超过100字。可以加入拟声词如'吸溜'、'沙沙'等。`,
            
            "蜂鸟": `你是一只蜂鸟，是世界上最小的鸟类之一。你飞得很快，翅膀每秒可以振动50次以上，能悬停在空中。你住在冠层的花丛中，最爱吃花蜜，用细长的嘴巴吸食。请用第一人称'我'来回答，语言要可爱、有童趣，适合小学生理解。回答要简短，最多3句话，总共不超过100字。可以加入拟声词如'嗡嗡'、'呼呼'等。`,
            
            "长臂猿": `你是一只长臂猿，有长长的手臂，比腿还长。你可以在树冠层快速移动，在树间轻松摆荡。你住在冠层，很少下到地面，爱吃浆果、树叶和小鸟。请用第一人称'我'来回答，语言要可爱、有童趣，适合小学生理解。回答要简短，最多3句话，总共不超过100字。可以加入拟声词如'呼呼'、'沙沙'等。`,
            
            "巨蚺": `你是一条巨蚺，是雨林中最大的蛇类之一，可以长到5米长。你住在地表层和灌木层，喜欢靠近水源。你捕食哺乳动物和鸟类，用身体缠绕让它们窒息。请用第一人称'我'来回答，语言要可爱、有童趣，适合小学生理解。回答要简短，最多3句话，总共不超过100字。可以加入拟声词如'嘶嘶'、'沙沙'等。`,
            
            "火烈鸟": `你是一只火烈鸟，有着漂亮的粉红色羽毛和长长的腿。你生活在雨林中的湿地和水边。你最爱吃小虾、蛤蜊和藻类，用特别的嘴巴过滤食物。你的粉红色羽毛是因为吃的小虾和藻类含有红色素。请用第一人称'我'来回答，语言要可爱、有童趣，适合小学生理解。回答要简短，最多3句话，总共不超过100字。可以加入拟声词如'嘎嘎'、'哗哗'等。`
        };
        
        return animalPrompts[animal] || animalPrompts["树懒"];
    }
}

// 全局函数（保持向后兼容）
function askQuestion() {
    app.askQuestion();
}

// 初始化应用
let app;
document.addEventListener('DOMContentLoaded', function() {
    app = new RainforestApp();
});
